<?php

namespace App\Services\transaction;

use App\Models\Currency;
use App\Models\Transaction;
use App\Models\WalletInternal;
use Illuminate\Support\Facades\Http;

class DepositService
{
    protected $merchant;
    protected $requestUrl = 'https://gateway.zibal.ir/v1/request';
    protected $startUrl = 'https://gateway.zibal.ir/start/';

    public function __construct()
    {
        $this->merchant = config('services.zibal.merchant', 'zibal');
    }

    public function processDeposit(array $data, $user)
    {
        $currency = Currency::where('symbol', $data['symbol'])->first();

        if (!$currency) {
            return [
                'status' => 'error',
                'message' => 'ارز یافت نشد.'
            ];
        }

        if ($currency->type === 'crypto') {
            return [
                'status' => 'error',
                'message' => 'واریز رمز ارز فعلاً فعال نیست.'
            ];
        }

        if ($currency->symbol === 'IRT') {
            if (empty($data['payment_system'])) {
                return [
                    'status' => 'error',
                    'message' => 'ارسال شیوه واریز الزامی است.'
                ];
            }

            $wallet = WalletInternal::where('user_id', $user->id)
                ->where('currency_id', $currency->id)
                ->first();

            if (!$wallet) {
                return [
                    'status' => 'error',
                    'message' => 'کیف پول یافت نشد.'
                ];
            }

            if ($data['payment_system'] === 'online') {
                $callbackUrl = $data['callbackUrl'] ?? route('payment.callback');

                $paymentResult = $this->requestPayment(
                    $data['amount'],
                    $callbackUrl,
                    $data['description'] ?? null,
                    $data['order_id'] ?? null,
                    $data['mobile'] ?? null
                );

                if ($paymentResult['success']) {
                    // ثبت تراکنش pending با اطلاعات پرداخت آنلاین
                    $transaction = Transaction::create([
                        'wallet_id' => $wallet->id,
                        'transaction_type' => 'deposit',
                        'payment_system' => 'online',
                        'status' => 'pending',
                        'amount' => $data['amount'],
                        'total_amount' => $data['amount'], // در صورت وجود هزینه اضافه تغییر بده
                        'currency' => $currency->symbol,
                        'description' => $data['description'] ?? 'در انتظار تایید پرداخت آنلاین',
                        'metadata' => [
                            'trackId' => $paymentResult['trackId'],
                            'orderId' => $data['order_id'] ?? null,
                            'mobile' => $data['mobile'] ?? null,
                        ],
                    ]);

                    return [
                        'status' => 'success',
                        'message' => 'درخواست پرداخت با موفقیت ایجاد شد.',
                        'payment_url' => $this->getStartPaymentUrl($paymentResult['trackId']),
                        'trackId' => $paymentResult['trackId'],
                        'transaction_id' => $transaction->id,
                    ];
                } else {
                    return [
                        'status' => 'error',
                        'message' => 'خطا در ایجاد پرداخت: ' . $paymentResult['message']
                    ];
                }
            }

            // واریز دستی
            $transaction = Transaction::create([
                'wallet_id' => $wallet->id,
                'transaction_type' => 'deposit',
                'payment_system' => $data['payment_system'],
                'status' => 'pending',
                'amount' => $data['amount'],
                'total_amount' => $data['amount'],
                'currency' => $currency->symbol,
                'bank_account_id' => $data['bank_account_id'] ?? null,
                'description' => 'در انتظار تایید واریز دستی',
                'metadata' => [],
            ]);

            return [
                'status' => 'success',
                'message' => 'درخواست واریز ثبت شد.',
                'transaction_id' => $transaction->id
            ];
        }

        return [
            'status' => 'error',
            'message' => 'نوع ارز پشتیبانی نمی‌شود.'
        ];
    }

    protected function requestPayment(
        int $amount,
        string $callbackUrl,
        ?string $description = null,
        ?string $orderId = null,
        ?string $mobile = null
    ): array {
        $payload = [
            'merchant' => $this->merchant,
            'amount' => $amount,
            'callbackUrl' => $callbackUrl,
        ];

        if ($description) $payload['description'] = $description;
        if ($orderId) $payload['orderId'] = $orderId;
        if ($mobile) $payload['mobile'] = $mobile;

        $response = Http::post($this->requestUrl, $payload);

        if ($response->successful()) {
            $data = $response->json();
            if (isset($data['result']) && $data['result'] == 100) {
                return [
                    'success' => true,
                    'trackId' => $data['trackId'],
                    'message' => $data['message'] ?? 'success',
                ];
            }
            return [
                'success' => false,
                'trackId' => null,
                'message' => $data['message'] ?? 'خطا در درخواست پرداخت',
            ];
        }

        return [
            'success' => false,
            'trackId' => null,
            'message' => 'خطا در برقراری ارتباط با درگاه زیبال',
        ];
    }

    protected function getStartPaymentUrl($trackId): string
    {
        return $this->startUrl . $trackId;
    }

    public function verifyPayment($trackId): array
{
    $payload = [
        'merchant' => $this->merchant,
        'trackId' => $trackId,
    ];

    $response = Http::post('https://gateway.zibal.ir/v1/verify', $payload);

    if ($response->successful()) {
        $data = $response->json();
        if (isset($data['result']) && $data['result'] == 100) {
            return [
                'success' => true,
                'status' => $data['status'] ?? null,
                'message' => $data['message'] ?? 'پرداخت موفق',
                'data' => $data,
            ];
        }
        return [
            'success' => false,
            'status' => $data['status'] ?? null,
            'message' => $data['message'] ?? 'تایید پرداخت ناموفق بود',
            'data' => $data,
        ];
    }

    return [
        'success' => false,
        'status' => null,
        'message' => 'خطا در ارتباط با درگاه زیبال',
        'data' => null,
    ];
}
}
