<?php

namespace App\Services\bank;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Http;
use App\Models\BankAccount;
use App\Models\User;

class BankAccountService
{
    protected $zibalToken;

    public function __construct()
    {
        $this->zibalToken = env('ZIBAL_API_TOKEN');
    }

    /**
     * ارسال درخواست به زبایل برای تطبیق کارت با کد ملی و تاریخ تولد
     */
    protected function checkCardWithNationalCode(string $nationalCode, string $birthDate, string $cardNumber): bool
    {
        $url = 'https://api.zibal.ir/v1/facility/checkCardWithNationalCode';

        $birthDateFormatted = str_replace('-', '/', $birthDate);
        $birthDateFormatted = strtok($birthDateFormatted, ' ');

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->zibalToken,
            'Accept' => 'application/json',
        ])->post($url, [
            'nationalCode' => $nationalCode,
            'birthDate' => $birthDateFormatted,
            'cardNumber' => $cardNumber,
        ]);

        if (!$response->successful()) {
            throw new \Exception('خطا در ارتباط با سرویس زبایل برای تطبیق کارت با کد ملی');
        }

        $data = $response->json();

        return isset($data['result'], $data['data']['matched'])
            && ($data['result'] == 1 || $data['result'] === true)
            && $data['data']['matched'] === true;
    }

    /**
     * گرفتن نام صاحب حساب، نام بانک و شماره حساب از API کارت به حساب
     */
    protected function getAccountInfoFromCard(string $cardNumber): array
    {
        $url = 'https://api.zibal.ir/v1/facility/cardToAccount';

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->zibalToken,
            'Accept' => 'application/json',
        ])->post($url, [
            'cardNumber' => $cardNumber,
        ]);

        if (!$response->successful()) {
            throw new \Exception('خطا در ارتباط با سرویس زبایل برای دریافت اطلاعات حساب از کارت');
        }

        $data = $response->json();

        if (isset($data['result'], $data['data']) && ($data['result'] == 1 || $data['result'] === true)) {
            return [
                'accountHolderName' => $data['data']['name'] ?? null,
                'bankName' => $data['data']['bankName'] ?? null,
                'accountNumber' => $data['data']['bankAccount'] ?? null,
            ];
        }

        throw new \Exception('دریافت اطلاعات حساب از کارت ناموفق بود.');
    }

    /**
     * تبدیل شماره کارت به شناسه شبا
     */
    protected function convertCardToSheba(string $cardNumber): ?string
    {
        $url = 'https://api.zibal.ir/v1/facility/cardToIban';

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->zibalToken,
            'Accept' => 'application/json',
        ])->post($url, [
            'cardNumber' => $cardNumber,
        ]);

        if (!$response->successful()) {
            throw new \Exception('خطا در ارتباط با سرویس زبایل برای تبدیل کارت به شبا');
        }

        $data = $response->json();

        if (isset($data['result']) && ($data['result'] == 1 || $data['result'] === true)) {
            if (!empty($data['data']['IBAN'])) {
                return $data['data']['IBAN'];  // این مقدار شبا هست
            }
        }

        return null;
    }

    public function handleBankAccount(array $data): array
{
    $userId = $data['user_id'];
    $currencyType = strtoupper($data['currency_type']);
    $cardNumber = $data['card_number'] ?? null;
    $iban = $data['iban'] ?? null;
    $bankName = $data['bank_name'] ?? null;
    $accountHolderName = $data['account_holder_name'] ?? null;

     $user = User::find($userId);
     if (!$user) {
      return [
        'status' => 'error',
        'message' => 'کاربر یافت نشد.'
        ];
      }

    $nationalCode = $user->national_code;
    $birthDate = $user->birthdate;

    // داده پایه برای ذخیره
    $accountData = [
        'card_number' => $cardNumber,
        'iban' => $iban,
        'bank_name' => $bankName,
        'account_holder_name' => $accountHolderName,
        'currency' => $currencyType,
    ];

    if ($currencyType === 'IRT') {
        if (!$cardNumber) {
            return [
                'status' => 'error',
                'message' => 'شماره کارت برای ارز IRT الزامی است.'
            ];
        }

        // بررسی تطبیق کارت با کد ملی و تاریخ تولد
        try {
            $matched = $this->checkCardWithNationalCode($nationalCode, $birthDate, $cardNumber);
        } catch (\Exception $e) {
            return [
                'status' => 'error',
                'message' => 'خطا در بررسی تطبیق کارت: ' . $e->getMessage(),
            ];
        }

        if (!$matched) {
            $accountData['status'] = 'inactive';

            // ذخیره رکورد جدید یا بروزرسانی رکورد مشابه (با شرط کارت و یوزر)
            BankAccount::updateOrCreate(
                ['user_id' => $userId, 'card_number' => $cardNumber],
                $accountData
            );

            return [
                'status' => 'error',
                'message' => 'عدم تطبیق کارت ملی با شماره کارت.',
            ];
        }

        // دریافت اطلاعات کامل حساب از کارت
        try {
            $accountInfo = $this->getAccountInfoFromCard($cardNumber);
            $accountData['account_holder_name'] = $accountInfo['accountHolderName'];
            $accountData['bank_name'] = $accountInfo['bankName'];
            $accountData['bank_account'] = $accountInfo['accountNumber'];
        } catch (\Exception $e) {
            // نادیده گرفتن خطا و ادامه روند
        }

        // تبدیل کارت به شبا
        try {
            $sheba = $this->convertCardToSheba($cardNumber);
            $accountData['sheba'] = $sheba;
            $accountData['status'] = 'active';
        } catch (\Exception $e) {
            return [
                'status' => 'error',
                'message' => 'خطا در تبدیل کارت به شبا: ' . $e->getMessage(),
            ];
        }

        if (!$accountData['sheba']) {
            return [
                'status' => 'error',
                'message' => 'تبدیل کارت به شبا ناموفق بود.',
            ];
        }

        // جلوگیری از ذخیره تکراری: اگر کارت یا شبا مشابهی موجود باشد، ذخیره نکن
        $exists = BankAccount::where('user_id', $userId)
            ->where(function ($query) use ($cardNumber, $accountData) {
                $query->where('card_number', $cardNumber)
                      ->orWhere('sheba', $accountData['sheba'] ?? '');
            })->exists();

        if ($exists) {
            return [
                'status' => 'error',
                'message' => 'این کارت یا شبا قبلا ثبت شده است.',
            ];
        }

        // ذخیره رکورد جدید
        BankAccount::create(array_merge($accountData, ['user_id' => $userId]));

        return [
            'status' => 'success',
            'data' => $accountData,
            'message' => 'اطلاعات حساب با موفقیت ثبت شد.',
        ];
    } else {
        // برای ارز غیر IRT، اطلاعات کامل و با وضعیت inactive ذخیره می شود
        if (!$iban || !$bankName || !$accountHolderName) {
            return [
                'status' => 'error',
                'message' => 'برای ارز غیر IRT باید شناسه IBAN، نام بانک و نام صاحب حساب ارسال شود.'
            ];
        }

        $accountData['status'] = 'inactive';

        // جلوگیری از ذخیره تکراری: اگر IBAN یا شماره کارت مشابه باشد ذخیره نکن
        $exists = BankAccount::where('user_id', $userId)
            ->where(function ($query) use ($iban, $cardNumber) {
                $query->where('iban', $iban);
                if ($cardNumber) {
                    $query->orWhere('card_number', $cardNumber);
                }
            })->exists();

        if ($exists) {
            return [
                'status' => 'error',
                'message' => 'این حساب بانکی قبلا ثبت شده است.',
            ];
        }

        // ذخیره رکورد جدید
        BankAccount::create(array_merge($accountData, ['user_id' => $userId]));

        return [
            'status' => 'success',
            'data' => $accountData,
            'message' => 'اطلاعات حساب با موفقیت ثبت شد.'
        ];
    }
}

    public function getUserBankAccounts(int $userId, string $currencyType): Collection
    {
        $query = BankAccount::where('user_id', $userId);
        switch (strtoupper($currencyType)) {
            case 'IRT':
                $query->where('currency', 'IRT');
                break;

            case 'USD':
                $query->where('currency', 'USD');
                break;

            case 'GLOBAL':
                $query->where('currency', '!=', 'IRT');
                break;

            case 'ALL':
                // هیچ فیلتری روی currency نمی‌ذاریم
                break;

            default:
                $query->where('currency', $currencyType); // حالت fallback
        }

        return $query->get([
            'id',
            'bank_name',
            'card_number',
            'bank_account',
            'sheba',
            'iban',
            'account_holder_name',
            'currency',
            'status',
            'activated_at',
            'deactivated_at',
        ]);
    }

}
