<?php

namespace App\Jobs;

use App\Models\Currency;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class UpdateCryptoPricesJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected int $chunkSize = 100;

    public function __construct()
    {
        // در صورت نیاز متغیرها را اینجا تعریف کن
    }

    public function handle()
    {
        Log::info('Start UpdateCryptoPricesJob');

        $currenciesQuery = Currency::where('is_active', true)
            ->whereNotNull('symbol')
            ->where('symbol', '!=', '');

        $currenciesQuery->chunk($this->chunkSize, function ($currencies) {

            $symbols = $currencies->pluck('symbol')->implode(',');

            Log::info("Updating prices for symbols chunk: {$symbols}");

            try {
                $response = Http::timeout(10)->get('https://min-api.cryptocompare.com/data/pricemultifull', [
                    'fsyms' => $symbols,
                    'tsyms' => 'USD',
                ]);
            } catch (\Exception $e) {
                Log::error('CryptoCompare API request failed: ' . $e->getMessage());
                return;
            }

            if (!$response->ok()) {
                Log::warning('CryptoCompare API returned status ' . $response->status());
                return;
            }

            $data = $response->json();

            if (!isset($data['RAW']) || empty($data['RAW'])) {
                Log::warning('CryptoCompare API response missing RAW data');
                return;
            }

            DB::transaction(function () use ($currencies, $data) {
                foreach ($currencies as $currency) {
                    $symbol = $currency->symbol;

                    if (isset($data['RAW'][$symbol]['USD'])) {
                        $priceData = $data['RAW'][$symbol]['USD'];

                        $currency->update([
                            'current_price' => $priceData['PRICE'] ?? 0,
                            'price_change_24h' => $priceData['CHANGEPCT24HOUR'] ?? 0,
                            'price_low_24h' => $priceData['LOW24HOUR'] ?? 0,
                            'price_high_24h' => $priceData['HIGH24HOUR'] ?? 0,
                            'volume_24h' => $priceData['VOLUME24HOURTO'] ?? 0,
                            'market_cap' => $priceData['MKTCAP'] ?? 0,
                            'price_change_1h' => $priceData['CHANGEPCTHOUR'] ?? null,
                            'price_low_1h' => $priceData['LOWHOUR'] ?? null,
                            'price_high_1h' => $priceData['HIGHHOUR'] ?? null,
                            'circulating_supply' => $priceData['CIRCULATINGSUPPLY'] ?? null,
                            'total_supply' => $priceData['TOTALSUPPLY'] ?? null,
                            'update_error' => null,
                            'updated_at' => now(),
                        ]);

                        Log::info("Updated price for symbol: {$symbol}");
                    } else {
                        $currency->update([
                            'update_error' => "Price data not found in API response at " . now()->toDateTimeString(),
                            'updated_at' => now(),
                        ]);
                        Log::warning("Price data missing for symbol: {$symbol}");
                    }
                }
            });
        });

        Log::info('Finished UpdateCryptoPricesJob');
    }
}
